/*-
 * Copyright (C)2010..2025 @BABOLO http://www.babolo.ru/
 * PKG = babolo-libmake
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*    */
static const char copyright[] = "\
@(#)Copyright (C)2010..2025 @BABOLO http://www.babolo.ru/\n\
@(#)All rights reserved.\n\
@(#)V.M "VERS;
static const char rcsid[] = "$Id: blin.c,v 1.93 2025/04/22 03:13:40 babolo Exp $";

#define BLIN_COMPAT VMAJOR
#define BLIN_INTERNAL
#ifdef BLIN_OSLinux
# define _GNU_SOURCE
#endif  /* BLIN_OSLinux */

#include <sys/types.h>
#include <sys/file.h>
#include <sysexits.h>
#include <sys/uio.h>
#include <strings.h>
#include <stddef.h>
#include <string.h>
#include <syslog.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stddef.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <err.h>
#include "../BLINflag.h"

#define ERROUT(A, B)    do {ex = (A); errno = (B); goto out;} while(0)

blin_pamparam blin_pamparam_internal =
{ 0x00000000
, 0
, { {NULL, BLIN_MODWARN | BLIN_MODEOAU, LOG_LOCAL7 | LOG_ERR    }
  , {NULL, BLIN_MODWARN | BLIN_MODEOAU, LOG_LOCAL7 | LOG_WARNING}
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  , {NULL, BLIN_MODPRNT | BLIN_MODEOAU, LOG_LOCAL7 | LOG_DEBUG  }
  }
, -1
#if BLINAP > 4
, 0
#endif
, 0
, 0
, 0
};

__weak_reference(blin_pamparam_internal, blin_internal_flags);
__weak_reference(blin_pamparam_internal, blin_direct_flags);

static const char * const
fac[] =
{ "KERN"  , "USER"  , "MAIL"    , "DAEMON", "AUTH"  , "SYSLOG"  , "LPR"    , "NEWS"
, "UUCP"  , "CRON"  , "AUTHPRIV", "FTP"   , "NTP"   , "SECURITY", "CONSOLE", "reserved"
, "LOCAL0", "LOCAL1", "LOCAL2"  , "LOCAL3", "LOCAL4", "LOCAL5"  , "LOCAL6" , "LOCAL7"
};

static const char * const
pri[] = {"EMERG", "ALERT", "CRIT", "ERR", "WARNING", "NOTICE", "INFO", "DEBUG"};

static const char * const
blinf[] = 
{ "MAININI", NULL  , NULL  , NULL  , NULL  , NULL  , NULL  , NULL
, NULL     , NULL  , NULL  , NULL  , NULL  , NULL  , NULL  , NULL
, NULL     , NULL  , NULL  , NULL  , NULL  , NULL  , NULL  , "TRAC"
, "BIT0"   , "BIT1", "BIT2", "BIT3", "BIT4", "BIT5", "BIT6", "BIT7"
};

static const char * const
whatf[] = 
{ NULL     , NULL     , NULL     , NULL  , "MODDTXT", NULL     , NULL     , NULL
, NULL     , NULL     , NULL     , NULL  , "MODDSTR", "MODDNON", "MODDREL", "MODTIME"
, "MODEOLF", "MODEOCR", "MODEOAU", NULL  , NULL     , "MODLOGR", "MODPRNT", "MODWARN"
, NULL     , NULL     , NULL     , NULL  , NULL     , NULL     , NULL     , NULL
};

#ifdef BLIN_OSLinux

# include <stdio_ext.h>

const int sys_nerr = 134;
const char * const
sys_errlist[] =
{ "Success"
, "Operation not permitted"
, "No such file or directory"
, "No such process"
, "Interrupted system call"
, "Input/output error"
, "No such device or address"
, "Argument list too long"
, "Exec format error"
, "Bad file descriptor"
, "No child processes"
, "Resource temporarily unavailable"
, "Cannot allocate memory"
, "Permission denied"
, "Bad address"
, "Block device required"
, "Device or resource busy"
, "File exists"
, "Invalid cross-device link"
, "No such device"
, "Not a directory"
, "Is a directory"
, "Invalid argument"
, "Too many open files in system"
, "Too many open files"
, "Inappropriate ioctl for device"
, "Text file busy"
, "File too large"
, "No space left on device"
, "Illegal seek"
, "Read-only file system"
, "Too many links"
, "Broken pipe"
, "Numerical argument out of domain"
, "Numerical result out of range"
, "Resource deadlock avoided"
, "File name too long"
, "No locks available"
, "Function not implemented"
, "Directory not empty"
, "Too many levels of symbolic links"
, "Unknown error 41"
, "No message of desired type"
, "Identifier removed"
, "Channel number out of range"
, "Level 2 not synchronized"
, "Level 3 halted"
, "Level 3 reset"
, "Link number out of range"
, "Protocol driver not attached"
, "No CSI structure available"
, "Level 2 halted"
, "Invalid exchange"
, "Invalid request descriptor"
, "Exchange full"
, "No anode"
, "Invalid request code"
, "Invalid slot"
, "Unknown error 58"
, "Bad font file format"
, "Device not a stream"
, "No data available"
, "Timer expired"
, "Out of streams resources"
, "Machine is not on the network"
, "Package not installed"
, "Object is remote"
, "Link has been severed"
, "Advertise error"
, "Srmount error"
, "Communication error on send"
, "Protocol error"
, "Multihop attempted"
, "RFS specific error"
, "Bad message"
, "Value too large for defined data type"
, "Name not unique on network"
, "File descriptor in bad state"
, "Remote address changed"
, "Can not access a needed shared library"
, "Accessing a corrupted shared library"
, ".lib section in a.out corrupted"
, "Attempting to link in too many shared libraries"
, "Cannot exec a shared library directly"
, "Invalid or incomplete multibyte or wide character"
, "Interrupted system call should be restarted"
, "Streams pipe error"
, "Too many users"
, "Socket operation on non-socket"
, "Destination address required"
, "Message too long"
, "Protocol wrong type for socket"
, "Protocol not available"
, "Protocol not supported"
, "Socket type not supported"
, "Operation not supported"
, "Protocol family not supported"
, "Address family not supported by protocol"
, "Address already in use"
, "Cannot assign requested address"
, "Network is down"
, "Network is unreachable"
, "Network dropped connection on reset"
, "Software caused connection abort"
, "Connection reset by peer"
, "No buffer space available"
, "Transport endpoint is already connected"
, "Transport endpoint is not connected"
, "Cannot send after transport endpoint shutdown"
, "Too many references: cannot splice"
, "Connection timed out"
, "Connection refused"
, "Host is down"
, "No route to host"
, "Operation already in progress"
, "Operation now in progress"
, "Stale file handle"
, "Structure needs cleaning"
, "Not a XENIX named type file"
, "No XENIX semaphores available"
, "Is a named type file"
, "Remote I/O error"
, "Disk quota exceeded"
, "No medium found"
, "Wrong medium type"
, "Operation canceled"
, "Required key not available"
, "Key has expired"
, "Key has been revoked"
, "Key was rejected by service"
, "Owner died"
, "State not recoverable"
, "Operation not possible due to RF-kill"
, "Memory page has hardware error"
};

const char *
getprogname(void) {
    return(program_invocation_short_name);
}

int
fpurge(FILE *stream) {
    if  (!!stream) __fpurge(stream);
    return(0);
}

void *
reallocf(void *ptr, size_t size) {
    void *r;

    if  (!(r = realloc(ptr, size))) free(ptr);
    return(r);
}

char *
strnstr(const char *s, const char *find, size_t slen) {
 /* Copyright (c) 2001 Mike Barcroft <mike@FreeBSD.org>
  * Copyright (c) 1990, 1993 The Regents of the University of California.  All rights reserved.
  * This code is derived from software contributed to Berkeley by Chris Torek.
  */
    size_t len;
    char   sc ;
    char   c  ;

    if  ((c = *find++) != '\0') {
        len = strlen(find);
        do {
            do {
                if  (slen-- < 1 || (sc = *s++) == '\0') return (NULL);
            } while (sc != c);
            if  (len > slen) return (NULL);
        } while (strncmp(s, find, len) != 0);
        s--;
    }
    return((char *)s);
}

size_t
strlcpy(char * __restrict dst, const char * __restrict src, size_t dsize) {
/* Copyright (c) 1998, 2015 Todd C. Miller <Todd.Miller@courtesan.com> */
    size_t      nleft = dsize;
    const char *osrc  = src;

    /* Copy as many bytes as will fit. */
    if  (nleft != 0) {
        while(--nleft != 0) {
            if  ((*dst++ = *src++) == '\0') break;
    }   }
    /* Not enough room in dst, add NUL and traverse rest of src. */
    if  (nleft == 0) {
        if  (dsize != 0) *dst = '\0'; /* NUL-terminate dst */
        while (*src++);
    }
    return(src - osrc - 1); /* count does not include NUL */
}

int
fdclose(FILE *fp, int *fdp) {
    int f1 = -1;
    int f2 = -1;

    f1 = fileno(fp);
    if  (0 <= f1) f2 = dup(f1);
    if  (!!fclose(fp)) f1 = -1;
    if  ((0 <= f1) && (0 <= f2)) {
        f1 = dup2(f2, f1);
        close(f2);
    }
    if  (!!fdp) *fdp = f1;
    return((0 > f1) ? EOF : 0);
}

FILE *
funopen( void    *cookie
       , int    (*readfn) (void *, char *, int)
       , int    (*writefn)(void *, const char *, int)
       , fpos_t (*seekfn) (void *, fpos_t, int)
       , int    (*closefn)(void *)
       ) {
    cookie_io_functions_t  a   ;
    const char            *mode;

    if  (!readfn) {
        mode = "a";
    } else if (!writefn) {
        mode = "r";
    } else {
        mode = "r+";
    }
    a.read  = (__ssize_t (*)(void *, char *, size_t))readfn ;
    a.write = (__ssize_t (*)(void *, const char *, size_t))writefn;
    a.seek  = (__ssize_t (*)(void *, __off64_t, int))seekfn;
    a.close = closefn;
    return(fopencookie(cookie, mode, a));
}

#endif /* BLIN_OSLinux */

#define SPC0 64
static const char *
spaces = "                                                                "; /* SPC0 * ' ' */

size_t
blin_internal_strnlen(const char *s, size_t maxlen) {
    size_t len;

    for (len = 0; len < maxlen; len++, s++) {
        if  (!*s) break;
    }
    return (len);
}
__weak_reference(blin_internal_strnlen, strnlen);

static void
/*****************************************************************************************************/
blin_init(void) {                                                                                /****
 *****************************************************************************************************/
    if  (!(blin_pamparam_internal.flags & BLIN_MAININI)) {
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: +blin_init %08X\n"
                   , blin_pamparam_internal.flags
                   )
            ;
        }
        for (int i = 0; i < 8; i++) blin_pamparam_internal.w[i].file = stderr;
        blin_pamparam_internal.flags |= BLIN_MAININI;
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: -blin_init %08X\n"
                   , blin_pamparam_internal.flags
                   )
            ;
}   }   }

BLIN_flag
/*****************************************************************************************************
 *****************************************************************************************************/
blin_doflag(u_int32_t mode, BLIN_flag w, BLIN_flag legal, BLIN_flag *f) {                        /****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t r = *f;              /*   */

    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf( blin_pamparam_internal.w[0].file
               , "BLIN: +blin_doflag %08X %08X %08X %08X\n"
               , mode, w, legal, r
               )
        ;
    }
    if  (mode & BLIN_CTL_RISE) {
        w = (((((mode & BLIN_CTL_NPRM) ? r : w) << 1) & BLIN_MASK) | BLIN_BIT0);
    } else if (mode & BLIN_CTL_NPRM) {
        w = 0;
    }
    if  (mode & BLIN_CTL_FNEG) w = ~w;
    w &= legal;
    switch (mode & (BLIN_CTL_NONE | BLIN_CTL_FAFM)) {
    case BLIN_CTL_FSET: *f |= w;                    break;
    case BLIN_CTL_FRST: *f &= ~w;                   break;
    case BLIN_CTL_FXOR: *f ^= w;                    break;
    case BLIN_CTL_FEQU: *f = (*f & ~legal) | w;     break;
    default: ;
    }
    if  (mode & BLIN_CTL_NPRE) r = *f;
    if  (mode & BLIN_CTL_RMSK) r &= legal;
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_doflag %08X\n", r);
    }
    return(r);
}

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_64(u_int64_t flags, const char * const *nm, int x, FILE *fo) {                              /****
 *****************************************************************************************************
 *****************************************************************************************************/
    int       ex = 0;
    u_int64_t m  ;
    int       i  ;
    int       k  ;
    int       l  ;

    if  (x > 64) x = 64;
    if  (x < -64) x = -64;
    l = (0 > x) ? x : -x;
    for (i = l; ~i; i >>= 1) l &= i;
    l = -l;
    switch(l) {
    case 0:
    case 1:
    case 2:
    case 4:
        if  (!(flags & 0xFFFFFFFFFFFFFFF0ULL)) {
            ex += fprintf(fo, "%01X", (u_int8_t)flags);
            break;
        }
        BLIN_FALLTHROUGH;
    case 8:
        if  (!(flags & 0xFFFFFFFFFFFFFF00ULL)) {
            ex += fprintf(fo, "%02X", (u_int8_t)flags);
            break;
        }
        BLIN_FALLTHROUGH;
    case 16:
        if  (!(flags & 0xFFFFFFFFFFFF0000ULL)) {
            ex += fprintf(fo, "%04X", (u_int16_t)flags);
            break;
        }
        BLIN_FALLTHROUGH;
    case 32:
        if  (!(flags & 0xFFFFFFFF00000000ULL)) {
            ex += fprintf(fo, "%08X", (u_int32_t)flags);
            break;
        }
        BLIN_FALLTHROUGH;
    default:
        ex += fprintf(fo, "%016"BLIN_O"X", flags);
    }
    k = 0;
    if  (0 > x) {
        for (i = 0, m = 1ULL << (l - 1); i < -x; ++i, m >>= 1) {
//printf("%d %016llX %016llX %016llX\n",i,m,flags,flags&m);
            if  ((flags & m) && !nm[i]) {
                ex += fprintf(fo, "%sunkn", k++ ? "," : "<");
                break;
        }   }
        for (i = 0, m = 1ULL << (l - 1); i < -x; ++i, m >>= 1) {
            if  ((flags & m) && !!nm[i]) {
                ex += fprintf(fo, "%s%s", k++ ? "," : "<", nm[i]);
        }   }
    } else if (0 < x) {
        for (i = 0, m = 1; i < x; ++i, m <<= 1) {
            if  ((flags & m) && !nm[i]) {
                ex += fprintf(fo, "%sunkn", k++ ? "," : "<");
                break;
        }   }
        for (i = 0, m = 1; i < x; ++i, m <<= 1) {
            if  ((flags & m) && !!nm[i]) {
                ex += fprintf(fo, "%s%s", k++ ? "," : "<", nm[i]);
    }   }   }
    if  (!!k) ex += fprintf(fo, ">");
    return(ex);
}

const void *
/*****************************************************************************************************
 *****************************************************************************************************/
blin_ctl(u_int32_t mode, ...) {                                                                  /****
 *****************************************************************************************************
 *****************************************************************************************************/
    const void *res = NULL;               /*   */
    va_list     ap  ;                     /*   va_arg */
    u_int32_t   m   ;                     /*  */
    u_int       i   ;                     /*    */

    va_start(ap, mode);
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: +blin_ctl %08X\n", mode);
    }
    blin_init();
    switch (mode & BLIN_CTL_OPER) {
    case BLIN_CTL_FLAG: {
            BLIN_flag w = va_arg(ap, BLIN_flag);

            res = (void *)(size_t)blin_doflag( mode
                                             , w
                                             , (BLIN_flag)BLIN_FLEGAL
                                             , &blin_pamparam_internal.flags
                                             );
        }
        break;
    case BLIN_CTL_WHAT: {
            BLIN_flag w = va_arg(ap, BLIN_flag);

            for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                if  (mode & m) {
                    res = (void *)(size_t)blin_doflag( mode
                                                     , w
                                                     , (BLIN_flag)~0
                                                     , &blin_pamparam_internal.w[i].what
                                                     );
        }   }   }
        break;
    case BLIN_CTL_FNAM: {
            char *f = va_arg(ap, char *);
            FILE *w = fopen(f, "a");

            if  (w == NULL) {
                fprintf(blin_pamparam_internal.w[0].file, "fopen=%s:%s\n", f, sys_errlist[errno]);
            } else {
                setvbuf(w, NULL, _IONBF, 0);
                for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                    if  (mode & m) {
                        res = blin_pamparam_internal.w[i].file;
                        if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.w[i].file = w;
                        if  (mode & BLIN_CTL_NPRE) res = blin_pamparam_internal.w[i].file;
        }   }   }   }
        break;
    case BLIN_CTL_FDES: {
            int   f = va_arg(ap, int);
            FILE *w = fdopen(f, "a");

            if  (w == NULL) {
                fprintf(blin_pamparam_internal.w[0].file, "fdopen=%d:%s\n", f, sys_errlist[errno]);
            } else {
                setvbuf(w, NULL, _IONBF, 0);
                for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                    if  (mode & m) {
                        res = blin_pamparam_internal.w[i].file;
                        if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.w[i].file = w;
                        if  (mode & BLIN_CTL_NPRE) res = blin_pamparam_internal.w[i].file;
        }   }   }   }
        break;
    case BLIN_CTL_FILE: {
            void *w = va_arg(ap, void *);

            for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                if  (mode & m) {
                    res = blin_pamparam_internal.w[i].file;
                    if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.w[i].file = w;
                    if  (mode & BLIN_CTL_NPRE) res = blin_pamparam_internal.w[i].file;
        }   }   }
        break;
    case BLIN_CTL_PRIO: {
            int32_t w = va_arg(ap, int32_t);

            for (m = BLIN_BIT0, i = 0; !!m; m <<= 1, ++i) {
                if  (mode & m) {
                    res = (void *)(ssize_t)(blin_pamparam_internal.w[i].prio);
                    if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.w[i].prio = w;
                    if  (mode & BLIN_CTL_NPRE) {
                        res = (void *)(ssize_t)(blin_pamparam_internal.w[i].prio);
        }   }   }   }
        break;
    case BLIN_CTL_TCFN: {
            char *f = va_arg(ap, char *);
            int   w = open(f, O_CREAT | O_WRONLY | O_TRUNC | O_APPEND, 0666);

            if  (w < 0) {
                fprintf( blin_pamparam_internal.w[0].file
                       , "%s:%s:%d: open(%s): %s\n"
                       , __FILE__
                       , __func__
                       , __LINE__
                       , f
                       , sys_errlist[errno]
                       )
                ;
            } else {
                res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
                if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.tracfile = w;
                if  (!(mode & BLIN_CTL_NPRE)) {
                    res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
        }   }   }
        break;
    case BLIN_CTL_TCFD: {
            int w = va_arg(ap, int);

            res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
            if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.tracfile = w;
            if  (!(mode & BLIN_CTL_NPRE)) {
                res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
        }   }
        break;
    case BLIN_CTL_TCFL: {
            FILE *f = va_arg(ap, void *);
            int   w = fileno(f);

            if  (w < 0) {
                fprintf( blin_pamparam_internal.w[0].file
                       , "%s:%s:%d: fileno: %s\n"
                       , __FILE__
                       , __func__
                       , __LINE__
                       , sys_errlist[errno]
                       )
                ;
            } else {
                res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
                if  (!(mode & BLIN_CTL_NONE)) blin_pamparam_internal.tracfile = w;
                if  (!(mode & BLIN_CTL_NPRE)) {
                    res = (void *)(ssize_t)(blin_pamparam_internal.tracfile);
        }   }   }
        break;
    case BLIN_CTL_CPRI:
        res = (const void *)copyright;
        break;
    case BLIN_CTL_RCSI:
        res = (const void *)rcsid;
        break;
    case BLIN_CTL_VERS:
        res = (const void *)VERS;
        break;
    case BLIN_CTL_DUMP: 
        fprintf( blin_pamparam_internal.w[0].file
               , "blin_pamparam_internal @ %"BLIN_X" "
               , BLIN_I(&blin_pamparam_internal)
               )
        ;
        blin_64(blin_pamparam_internal.flags, blinf, 32, blin_pamparam_internal.w[0].file);
        if  (blin_pamparam_internal.err) {
            fprintf( blin_pamparam_internal.w[0].file
        	   , " !!! %d !!!\n"
        	   , blin_pamparam_internal.err
        	   )
            ;
        } else {
            fprintf(blin_pamparam_internal.w[0].file, "\n");
        }
        for (i = 0; i < 8; ++i) {
#           define  SPC1 40
#           define  SPC2 24
            int     d;
            int32_t p;
            int     k;

            fprintf(blin_pamparam_internal.w[0].file, "  %u ", i);
            d = blin_64( blin_pamparam_internal.w[i].what
        	       , whatf
        	       , 32
        	       , blin_pamparam_internal.w[0].file
        	       )
            ;
            if  (SPC1 > d) {
        	fprintf(blin_pamparam_internal.w[0].file, "%s", &spaces[SPC0 - SPC1 + d]);
        	d = 0;
            } else {
        	d -= SPC1;
            }
            k = 0;
            p = blin_pamparam_internal.w[i].prio;
            d += fprintf(blin_pamparam_internal.w[0].file, " %08X", p);
            if  (p & ~(LOG_PRIMASK | LOG_FACMASK)) {
        	d += fprintf(blin_pamparam_internal.w[0].file, "%sunkn", k++ ? "," : "<");
            }
            if  (LOG_FAC(p) < LOG_NFACILITIES) {
        	d += fprintf( blin_pamparam_internal.w[0].file
        		    , "%s%s"
        		    , k++ ? "," : "<"
        		    , fac[(p & LOG_FACMASK) >> 3]
        		    )
        	;
            } else {
        	d += fprintf( blin_pamparam_internal.w[0].file
        		    , "%s*%02X*"
        		    , k++ ? "," : "<"
        		    , LOG_FAC(p)
        		    )
        	;
            }
            d += fprintf( blin_pamparam_internal.w[0].file
        		, "%s%s"
        		, k++ ? "," : "<"
        		, pri[p & LOG_PRIMASK]
        		)
            ;
            d += fprintf(blin_pamparam_internal.w[0].file, ">");
            if  (SPC2 > d) {
        	fprintf(blin_pamparam_internal.w[0].file, "%s", &spaces[SPC0 - SPC2 + d]);
            }
            if  (blin_pamparam_internal.w[i].file == stdin) {
        	fprintf(blin_pamparam_internal.w[0].file, " file=stdin\n");
            } else if (blin_pamparam_internal.w[i].file == stdout) {
        	fprintf(blin_pamparam_internal.w[0].file, " file=stdout\n");
            } else if (blin_pamparam_internal.w[i].file == stderr) {
        	fprintf(blin_pamparam_internal.w[0].file, " file=stderr\n");
            } else {
                fprintf( blin_pamparam_internal.w[0].file
                       , " file=%"BLIN_X"\n"
                       , BLIN_I(blin_pamparam_internal.w[i].file)
                       )
                ;
        }   }
        fprintf(blin_pamparam_internal.w[0].file, " track=%d\n", blin_pamparam_internal.tracfile);
        break;
    default: ;
    }
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_ctl %"BLIN_X"\n", BLIN_I(res));
    }
    va_end(ap);
    return(res);
}

static void
/*****************************************************************************************************/
doout(u_int level, char *text, const char *file, const char *func, int line, int8_t er) {        /****
 *****************************************************************************************************/
    BLIN_trace        *tracinf;
    const char * const prog   = getprogname();
    u_int32_t          ctl    = blin_pamparam_internal.w[level & 7].what;
    struct timespec    tp     ;                      /*      */

    clock_gettime(CLOCK_REALTIME, &tp);
    if  ((ctl & BLIN_MODWARN) && (level & BLIN_DOCRY)) {
        if  (ctl & BLIN_MODTIME) {
            warnx("%10"BLIN_D".%09ld %s:%s:%d: %s", tp.tv_sec, tp.tv_nsec, file, func, line, text);
        } else {
            warnx("%s:%s:%d: %s", file, func, line, text);
    }   }
    if  ((ctl & BLIN_MODPRNT) && (level & BLIN_DOCRY)) {
        if  (ctl & BLIN_MODTIME) {
            fprintf( blin_pamparam_internal.w[level & 7].file
                   , "%10"BLIN_D".%09ld %s:%s:%d: %s"
                   , tp.tv_sec, tp.tv_nsec, file, func, line, text
                   );
        } else {
            fprintf(blin_pamparam_internal.w[level & 7].file, "%s:%s:%d: %s", file, func, line, text);
    }   }
    if  ((ctl & BLIN_MODLOGR) && (level & BLIN_DOCRY)) {
        if  (ctl & BLIN_MODTIME) {
            syslog( blin_pamparam_internal.w[level & 7].prio
                  , "%10"BLIN_D".%09ld %s:%s:%d: %s"
                  , tp.tv_sec, tp.tv_nsec, file, func, line, text
                  );
        } else {
            syslog(blin_pamparam_internal.w[level & 7].prio, "%s:%s:%d: %s", file, func, line, text);
    }   }
    if  (blin_pamparam_internal.tracfile >= 0) {
        char              *textpoint;
        size_t             textlen  ;
        u_int8_t           progsz   = (u_int8_t)strnlen(prog, 255);
        u_int8_t           filesz   = (u_int8_t)strnlen(file, 255);
        u_int8_t           funcsz   = (u_int8_t)strnlen(func, 255);
        size_t             tmplen   ;

        textlen = (size_t)filesz + (size_t)funcsz + (size_t)progsz + strlen(text);
        tmplen = offsetof(BLIN_trace, string) + textlen;
        tmplen = (tmplen + BLIN_TALLIGN - 1) & ~((size_t)(BLIN_TALLIGN - 1));
        if  (!(tracinf = malloc(tmplen))) {
            goto out; /* XXXX errprint */
        }
        tracinf->len    = (u_int32_t)tmplen;
        bzero((char *)tracinf + tracinf->len - BLIN_TALLIGN, BLIN_TALLIGN);
        tracinf->erno   = er;
        tracinf->pid    = getpid();
        tracinf->time   = (u_int64_t)(tp.tv_sec * 1000000000L + tp.tv_nsec);
        tracinf->level  = level & (BLIN_TALLIGN - 1);
        tracinf->line   = (u_int32_t)line;
        tracinf->progsz = progsz;
        tracinf->filesz = filesz;
        tracinf->funcsz = funcsz;
        textpoint = tracinf->string;
        textpoint = stpncpy(textpoint, prog, (size_t)tracinf->progsz);
        textpoint = stpncpy(textpoint, file, (size_t)tracinf->filesz);
        textpoint = stpncpy(textpoint, func, (size_t)tracinf->funcsz);
        (void)stpncpy(textpoint, text, strlen(text));
        flock(blin_pamparam_internal.tracfile, LOCK_EX);
        write(blin_pamparam_internal.tracfile, tracinf, (size_t)tracinf->len);
        flock(blin_pamparam_internal.tracfile, LOCK_UN);
        free(tracinf);
    }
out:
    return;
}

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_cry4(int ctl, const char *file, const char *func, int nline, const char *format, ...) {     /****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t control;                      /* what    blin_pamparam_internal */
    ssize_t   avail;                        /*                              */
    ssize_t   need = 0;                     /*                                 */
    char     *buf  = NULL;                  /*                                                 */
    char     *ptr  = NULL;                  /*  */
    char      qq   [BLIN_MAXCRY1 + 4];      /*                                      */
    int       er   = errno;                 /*  errno                                      */
    int       ex   = EX_OK;                 /*                                              */
    va_list   ap   ;                        /*   va_arg                               */
    ssize_t   t    ;                        /*  */

    blin_init();
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: +blin_cry4 %08X\n", ctl);
    }
    control = blin_pamparam_internal.w[ctl & 7].what;
    need = 0;
    if  (!(ctl & BLIN_DOCRY) && (blin_pamparam_internal.tracfile < 0)) goto out;
    do {
        va_start(ap, format);
        if  (!need) {
            buf = qq;
            avail = BLIN_MAXCRY4;
        } else {
            buf = malloc((unsigned long)need + 5);
            avail = need + 1;
            need = 0;
            if  (!buf) {
                ifBLIN_QX0("malloc");
                ERROUT(EX_OSERR, ENOMEM);
        }   }
        ptr = buf;
        
        t = vsnprintf(ptr, (unsigned long)avail + 1, format, ap);
        if  (0 > t) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: %s:%s:%d: %s\n"
                   , __FILE__
                   , __func__
                   , __LINE__
                   , strerror(errno)
                   );
            goto out;
        }
        need += t;
        if  (0 >= (avail -= t)) avail = 0; else ptr += t;
        if  (ctl & 8) {
#           define UPREFIX         "Unknown error"
#           define EBUFSIZE        11U

            if  (avail-- > 0) *(ptr++) = ':';
            if  (avail-- > 0) *(ptr++) = ' ';
            need += 2;
            if  ((er < sys_nerr) && (er >= 0)) {
                for (t = 0; sys_errlist[er][t]; ++t) {
                    if  (avail > 0) {
                        *(ptr++) = sys_errlist[er][t];
                        --avail;
                    }
                    ++need;
                }
            } else if ((0 > avail) || ((size_t)avail < (EBUFSIZE + strlen(UPREFIX)))) {
                need += (ssize_t)(EBUFSIZE + strlen(UPREFIX));
                avail = 0;
                if  (er < 0) {
                    ++need;
                    --avail;
                }
            } else {
                char *e;
                int s;

                for (t = 0; UPREFIX[t]; ++t) {
                    if  (avail > 0) {
                        *(ptr++) = UPREFIX[t];
                        --avail;
                    }
                    ++need;
                }
                e = ptr + EBUFSIZE;
                need += EBUFSIZE;
                avail -= EBUFSIZE;
                if  (er < 0) {
                    s = -er;
                    ++e;
                    ++need;
                    --avail;
                } else {
                    s = er;
                }
                do *--e = "0123456789"[s % 10]; while (s /= 10);
                if  (er < 0) *--e = '-';
                while (e > ptr) *--e = ' ';
                ptr += EBUFSIZE;
                if  (er < 0) ++ptr;
        }   }
        if  (control & BLIN_MODEOAU) {
            if  ((ptr <= buf) || (ptr[-1] != '\n')) control |= BLIN_MODEOLF;
        }
        if  (!(~control & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(ptr++) = '\r';
            --avail;
            ++need;
        }
        if  (!(~control & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(ptr++) = '\n';
            --avail;
            ++need;
        }
        va_end(ap);
    } while (avail <= 0);
    *(ptr++) = '\0';
    doout(ctl & (BLIN_DOCRY | 7), buf, file, func, nline, (int8_t)er);
out:
    if  (buf != qq) free(buf);
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_cry4 %d %"BLIN_D"\n", ex, need);
    }
    blin_pamparam_internal.err = errno;
    errno = er;
    return(ex);
}

static void
/*****************************************************************************************************/
doout3(int level, char *q) {                                                                      /****
 *****************************************************************************************************/
    u_int32_t ctl = blin_pamparam_internal.w[level & 7].what;

    if  (ctl & BLIN_MODWARN) {
        warnx("%s", q);
    }
    if  (ctl & BLIN_MODPRNT) {
        fprintf(blin_pamparam_internal.w[level & 7].file, "%s", q);
    }
    if  (ctl & BLIN_MODLOGR) {
        syslog(blin_pamparam_internal.w[level & 7].prio, "%s", q);
}   }

static void
/*****************************************************************************************************/
dotime(int level) {                                                                              /****
 *****************************************************************************************************/
    u_int32_t       ctl = blin_pamparam_internal.w[level].what;
    struct timespec tp  ;                            /*      */

    if  (ctl & BLIN_MODTIME) {
        if  (0 <= clock_gettime(CLOCK_REALTIME, &tp)) {
            if  (ctl & BLIN_MODWARN) {
                warnx("%10"BLIN_D".%09ld", tp.tv_sec, tp.tv_nsec);
            }
            if  (ctl & BLIN_MODPRNT) {
                fprintf( blin_pamparam_internal.w[level].file, "%10"BLIN_D".%09ld\n"
                       , tp.tv_sec, tp.tv_nsec
                       );
            }
            if  (ctl & BLIN_MODLOGR) {
                syslog( blin_pamparam_internal.w[level].prio, "%10"BLIN_D".%09ld"
                      , tp.tv_sec, tp.tv_nsec
                      );
}   }   }   }

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_stateheader(blin_statectl staflag, const char actnames[][3]) {                              /****
 *****************************************************************************************************
 *****************************************************************************************************/
#if BLINAP > 4
    const char *header = "nst rs cl st --rg32-- ----offs----";
#else
    const char *header = "nst rs cl st --rg32-- --offs--";
#endif
    u_int32_t   ctl    ;
    int         ex     = EX_OK;
    char       *q      ;
    char       *p      ;
    int         i      ;
    ssize_t     l      ;
    ssize_t     t      ;

    blin_init();
    if  (blin_pamparam_internal.flags & ((BLIN_flag)BLIN_BIT0 << staflag.level)) {
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: +blin_stateheader %08X\n"
                   , staflag.statectl
                   )
            ;
        }
        dotime(staflag.level);
        ctl = blin_pamparam_internal.w[staflag.level].what;
        l = strlen("control-");
        if  (!staflag.allign) {
            for (i = 0; i < 24; ++i) if (actnames[i][0] != '-') l += 3;
        } else {
            l += 3 * staflag.allign;
        }
        l += (ssize_t)strlen(header);
        l += 3; /* for \r\n\0 */
            /*     "---"   */
        if  (!(q = calloc(1, (unsigned long)l + 3))) {
            ifBLIN_QX0("malloc");
            ERROUT(EX_OSERR, ENOMEM);
        }
        t = l;
        strncpy(q, "control-", (unsigned long)t);
            /*  q  '\0'  strncpy()   calloc() */
        p = index(q, 0);
        if  (0 >= (t = q + l - p)) {
            ifBLIN_QX0("strcat #0");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (!staflag.allign) {
            for (i = 0; i < 24; ++i) {
                if  (actnames[i][0] != '-') {
                    *(p++) = ' ';
                    *(p++) = actnames[i][0];
                    *(p++) = actnames[i][1];
                    if  (0 >= (t -= 3)) {
                        ifBLIN_QX0("strcat #1 %d", i);
                        ERROUT(EX_SOFTWARE, EDOOFUS);
            }   }   }
        } else  {
            for (i = 0; i < staflag.allign; ++i) {
                *(p++) = ' ';
                *(p++) = '-';
                *(p++) = '-';
                if  (0 >= (t -= 3)) {
                    ifBLIN_QX0("strcat #2 %d", i);
                    ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }   }
            /*    (++t)      */
        strncat(q, header, (unsigned long)++t);
        p = index(q, 0);
        if  (0 > (t = q + l - p)) {
            ifBLIN_QX0("strcat #3 real %"BLIN_U" > %"BLIN_U" alloc", BLIN_I(p - q), BLIN_I(l));
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (ctl & BLIN_MODEOAU) {
            if  (p[-1] != '\n') ctl |= BLIN_MODEOLF;
        }
        if  (!(~ctl & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(p++) = '\r';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #4");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (!(~ctl & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(p++) = '\n';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #5");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        *(p++) = '\0';
        doout3(staflag.level, q);
out:
        free(q);
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_stateheader %d\n", ex);
    }   }
    return(ex);
}

int
/*****************************************************************************************************/
/*****************************************************************************************************/
blin_statebody( blin_statectl  staflag         /* level gtequ staten actinn bordsy allign revert     */
              , const char     actnames[][3]   /*                               */
              , const char     statenames[][3] /*                              */
              , const char     clasnname[3]    /*                            */
              , const char    *in              /*                        */
              , size_t         offset          /* 32/48 , ,    */
              , u_int64_t      control         /*                            */
              , u_int32_t      state           /*                                   */
              , u_int32_t      reg             /*   32                  */
              , u_char         byte            /*   8                   */
              ) {                                                                                   /*
 *****************************************************************************************************/
/*****************************************************************************************************/
    u_int64_t      contrana;
    u_int64_t      C_state = BLIN_MIMASK(staflag.staten);
    u_int64_t      flymask ;
    u_int32_t      ctl     ;
    int            ex      = EX_OK;
    u_int          i       ;
    int            k       ;
    ssize_t        l       ;
    ssize_t        t       ;
    char          *q       ;
    char          *p       ;

    blin_init();
    if  (blin_pamparam_internal.flags & ((BLIN_flag)BLIN_BIT0 << staflag.level)) {
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf( blin_pamparam_internal.w[0].file
                   , "BLIN: +blin_statebody %08X\n"
                   , staflag.statectl
                   )
            ;
        }
        if  (staflag.revert) {
            flymask = (u_int64_t)1 << (staflag.staten + staflag.actinn - 1);
        } else {
            flymask = (u_int64_t)1 << (staflag.staten);
        }
        ctl = blin_pamparam_internal.w[staflag.level].what;
        l = 9;
        if  (!staflag.allign) {
            for (i = 0; i < 24; ++i) if (actnames[i][0] != '-') l += 3;
        } else {
            l += 3 * staflag.allign;
        }
        l += 3 + 3 + 3 + 3 + 9;
#       if BLINAP > 4
        l += 13;
#       else
        l += 9;
#       endif
        l += BLIN_MAXTXTSTA + 3;
        l += 3; /* for \r\n\0 */
        if  (!(q = calloc(1, (unsigned long)l + 3))) {
            ifBLIN_QX0("malloc");
            ERROUT(EX_OSERR, ENOMEM);
        }
        p = q + snprintf(q, (unsigned long)l, "%08X ", (u_int32_t)control);
        if  (0 >= (t = q + l - p)) {
            ifBLIN_QX0("strcat #0");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (!staflag.allign) {
            for (i = 0, contrana = control, k = 0; i < staflag.actinn; ++i) {
                if  (actnames[i][0] == '-') {
                    if  (contrana & flymask) {
                        ++k;
                    }
                } else {
                    if  (contrana & flymask) {
                        *(p++) = actnames[i][0];
                        *(p++) = actnames[i][1];
                    } else {
                        *(p++) = ' ';
                        *(p++) = ' ';
                    }
                    if  (k && ((i + 1) == staflag.actinn)) {
                        *(p++) = '-';
                    } else {
                        *(p++) = ' ';
                    }
                    if  (0 >= (t -= 3)) {
                        ifBLIN_QX0("strcat #1 %u", i);
                        ERROUT(EX_SOFTWARE, EDOOFUS);
                }   }
                if  (staflag.revert) contrana <<= 1; else contrana >>= 1;
            }
        } else {
            for (i = 0, contrana = control, k = 0; i < staflag.actinn; ++i) {
                if  (contrana & flymask) {
                    *(p++) = actnames[i][0];
                    *(p++) = actnames[i][1];
                    *(p++) = ' ';
                    if  (0 >= (t -= 3)) {
                        ifBLIN_QX0("strcat #1 %u", i);
                        ERROUT(EX_SOFTWARE, EDOOFUS);
                    }
                    k++;
                }
                if  (staflag.revert) contrana <<= 1; else contrana >>= 1;
            }
            for (; k < staflag.allign; ++k) {
                *(p++) = ' ';
                *(p++) = ' ';
                *(p++) = ' ';
                if  (0 >= (t -= 3)) {
                    ifBLIN_QX0("strcat #2 %u", i);
                    ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }   }
        p += snprintf( p, (unsigned long)t, "%s %02X %s %s %08X %"BLIN_X" ~"
                     , statenames[control & C_state]
                     , byte & 0x00FF
                     , clasnname
                     , statenames[state]
                     , reg
                     , BLIN_I(offset)
                     );
        if  (0 >= (t = q + l - p)) {
            ifBLIN_QX0("strcat #3");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        for (i = 0; i <= BLIN_MAXTXTSTA; ++i) {
            char c;

            if  (staflag.offmax && (i >= offset)) break;
            c = in[i + (staflag.offsus ? offset : 0)];
            if  (!c) break;
            if  (staflag.gtequ ? ((c & 0xFF) < staflag.bordsy) : ((c & 0xFF) == staflag.bordsy)) break;
            *(p++) = c;
            if  (0 > (--t)) {
                ifBLIN_QX0("strcat #4");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (i <= BLIN_MAXTXTSTA) {
            *(p++) = '~';
            --t;
        } else if (0 > t) {
            ifBLIN_QX0("strcat #5");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (ctl & BLIN_MODEOAU) {
            if  (p[-1] != '\n') ctl |= BLIN_MODEOLF;
        }
        if  (!(~ctl & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(p++) = '\r';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #6");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (!(~ctl & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(p++) = '\n';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #7");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        *(p++) = '\0';
        doout3(staflag.level, q);
out:
        free(q);
        if  (blin_pamparam_internal.flags & BLIN_TRAC) {
            fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_statebody %d\n", ex);
    }   }
    return(ex);
}

static char
z(u_char u) {
    u_char s;

    if  (u < ' ') s = '.';
    else if ((u < 0x7F) || (u >= (u_char)'') || (u == (u_char)'') || (u == (u_char)'')) s = u;
    else s = '.';
    return((char)s);
}

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_dumb(int mode, const void *buf, ssize_t len, ...) {                                         /****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t     control;
    int           ex = EX_OK;
    va_list       ap;
    char          q    [BLIN_MAXTXTDUM + BLIN_MAXSTRDUM + 3];
    char         *p;
    char         *s;
    int           i;
    const u_char *c;
    ssize_t       t;
    size_t        l;

    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: +blin_dumb %08X %"BLIN_D"\n", mode, len);
    }
    c = buf;
    dotime(mode & 7);
    control = blin_pamparam_internal.w[mode & 7].what;
    while (len > 0) {
        p = q;
        if  (control & BLIN_MODDSTR) {
            va_start(ap, len);
            s = va_arg(ap, char *);
            if  (BLIN_MAXSTRDUM < (l = strnlen(s, BLIN_MAXSTRDUM + 3))) {
                ifBLIN_QX0("Long label %zu > %u", l, BLIN_MAXSTRDUM);
                ERROUT(EX_USAGE, EMSGSIZE);
            }
            bcopy(s, p, l);
            p += l;
            va_end(ap);
        }
        if  (0 >= (t = q + BLIN_MAXTXTDUM - p)) {
            ifBLIN_QX0("strcat label");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        if  (control & BLIN_MODDNON) {
        } else if (control & BLIN_MODDREL) {
            p += snprintf(p, BLIN_MAXTXTDUM, "%04X", (int)(c - (const u_char*)buf));
        } else {
            p += snprintf(p, BLIN_MAXTXTDUM, "%"BLIN_X, BLIN_I(c));
        }
        if  (0 >= (t = q + BLIN_MAXTXTDUM - p)) {
            ifBLIN_QX0("strcat #0");
            ERROUT(EX_SOFTWARE, EDOOFUS);
        }
        for (i = 0; i < 16; ++i) {
            if  (i && !(i & 3)) {
                *(p++) = ' ';
                t--;
                if  (0 > t) {
                    ifBLIN_QX0("strcat #1 %d", i);
                    ERROUT(EX_SOFTWARE, EDOOFUS);
            }   }
            if  (i < len) {
                p += snprintf(p, (unsigned long)t, " %02X", c[i]);
            } else if ((mode & BLIN_MODDTXT) || (control & BLIN_MODDTXT)) {
                *(p++) = ' ';
                *(p++) = ' ';
                *(p++) = ' ';
            }
            if  (0 >= (t = q + BLIN_MAXTXTDUM - p)) {
                ifBLIN_QX0("strcat #2 %d", i);
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  ((mode & BLIN_MODDTXT) || (control & BLIN_MODDTXT)) {
            *(p++) = ' ';
            *(p++) = '|';
            if  (0 >= (t -= 2)) {
                ifBLIN_QX0("strcat #3");
                ERROUT(EX_SOFTWARE, EDOOFUS);
            }
            for (i = 0; i < 16; ++i) {
                if  (i < len) {
                    *(p++) = z(c[i]);
                    if  (0 >= (--t)) {
                        ifBLIN_QX0("strcat #4");
                        ERROUT(EX_SOFTWARE, EDOOFUS);
            }   }   }
            *(p++) = '|';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #5");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (control & BLIN_MODEOAU) control |= BLIN_MODEOLF;
        if  (!(~control & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(p++) = '\r';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #6");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        if  (!(~control & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(p++) = '\n';
            if  (0 >= (--t)) {
                ifBLIN_QX0("strcat #7");
                ERROUT(EX_SOFTWARE, EDOOFUS);
        }   }
        *(p++) = '\0';
        --t;
        doout3(mode & 7, q);
        if  (len >= 16) {
            c += 16;
            len -= 16;
        } else {
            c += len;
            len = 0;
    }   }
out:
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_dumb %d\n", ex);
    }
    return(ex);
}

/*****************************************************************************************************
 *****************************************************************************************************
 **************************************     V3  **************************************
 *****************************************************************************************************
 *****************************************************************************************************/

int
/*****************************************************************************************************
 *****************************************************************************************************/
blin_cry1(int ctl, const char *format, ...) {                                                    /****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t       control;                /* what    blin_pamparam_internal */
    ssize_t         avail;                  /*                              */
    ssize_t         need = 0;               /*                                 */
    char           *buf;                    /*                                                 */
    char           *ptr;                    /*  */
    char            qq   [BLIN_MAXCRY1 + 4];/*                                      */
    struct timespec tp;                     /*               */
    int             er;                     /*  errno                                      */
    int             ex   = EX_OK;           /*                                              */
    va_list         ap;                     /*   va_arg                               */
    ssize_t         t;                      /*  */

    er = errno;
    blin_init();
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: +blin_cry1 %08X\n", ctl);
    }
    control = blin_pamparam_internal.w[ctl & 7].what;
    if  (control & BLIN_MODTIME) {
        if  (0 > clock_gettime(CLOCK_REALTIME, &tp)) control &= ~(u_int32_t)BLIN_MODTIME;
    }
    need = 0;
    do {
        va_start(ap, format);
        if  (!need) {
            buf = qq;
            avail = BLIN_MAXCRY1;
        } else {
            buf = malloc((unsigned int)need + 5);
            avail = need + 1;
            need = 0;
            if  (!buf) {
                ifBLIN_QX0("malloc");
                ERROUT(EX_OSERR, ENOMEM);
        }   }
        ptr = buf;
        if  (control & BLIN_MODTIME) {
            t = snprintf(ptr, (unsigned long)avail + 1, "%10"BLIN_D".%09ld ", tp.tv_sec, tp.tv_nsec);
            need += t;
            if  (0 > (avail -= t)) avail = 0; else ptr += t;
        }
        t = vsnprintf(ptr, (unsigned long)avail + 1, format, ap);
        need += t;
        if  (0 >= (avail -= t)) avail = 0; else ptr += t;
        if  (ctl & 8) {
#           define UPREFIX         "Unknown error"

            if  (avail-- > 0) *(ptr++) = ':';
            if  (avail-- > 0) *(ptr++) = ' ';
            need += 2;
            if  ((er < sys_nerr) && (er >= 0)) {
                for (t = 0; sys_errlist[er][t]; ++t) {
                    if  (avail > 0) {
                        *(ptr++) = sys_errlist[er][t];
                        --avail;
                    }
                    ++need;
                }
            } else if ((0 > avail) || ((size_t)avail < (EBUFSIZE + strlen(UPREFIX)))) {
                need += (ssize_t)(EBUFSIZE + strlen(UPREFIX));
                avail = 0;
                if  (er < 0) {
                    ++need;
                    --avail;
                }
            } else {
                char *e;
                int s;

                for (t = 0; UPREFIX[t]; ++t) {
                    if  (avail > 0) {
                        *(ptr++) = UPREFIX[t];
                        --avail;
                    }
                    ++need;
                }
                e = ptr + EBUFSIZE;
                need += EBUFSIZE;
                avail -= EBUFSIZE;
                if  (er < 0) {
                    s = -er;
                    ++e;
                    ++need;
                    --avail;
                } else {
                    s = er;
                }
                do *--e = "0123456789"[s % 10]; while (s /= 10);
                if (er < 0) *--e = '-';
                while (e > ptr) *--e = ' ';
                ptr += EBUFSIZE;
                if  (er < 0) ++ptr;
        }   }
        if  (control & BLIN_MODEOAU) {
            if  (ptr[-1] != '\n') control |= BLIN_MODEOLF;
        }
        if  (!(~control & (BLIN_MODEOCR | BLIN_MODPRNT))) {
            *(ptr++) = '\r';
            --avail;
            ++need;
        }
        if  (!(~control & (BLIN_MODEOLF | BLIN_MODPRNT))) {
            *(ptr++) = '\n';
            --avail;
            ++need;
        }
        va_end(ap);
    } while (avail <= 0);
    *(ptr++) = '\0';
    doout3(ctl & 7, buf);
out:
    if  (buf != qq) free(buf);
    if  (blin_pamparam_internal.flags & BLIN_TRAC) {
        fprintf(blin_pamparam_internal.w[0].file, "BLIN: -blin_cry1 %d %"BLIN_D"\n", ex, need);
    }
    blin_pamparam_internal.err = errno;
    errno = er;
    return(ex);
}
