/*-
 * Copyright (C)2013..2025 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2013..2025 @BABOLO http://www.babolo.ru/"
#ident "@(#) $Id: json2pg.c,v 1.25 2025/02/03 13:30:26 babolo Exp $"

#define BLIN_COMPAT  4
#define MIFE_COMPAT  5
#define Bpars_COMPAT 4

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <errno.h>
#include <err.h>
#include <json-c/json.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <getCGIparm.h>
#include <mife.h>

#define JSON2PG_TYNUM 7

typedef struct json2pg_path json2pg_path;
struct json2pg_path {
    json2pg_path  *next ;
    char         **path ;
};

typedef struct {
    BLIN_flag     flags   ;
#   define JSON2PG_DUMP    0x000100
#   define JSON2PG_ODMP /* 0x000200 */ (JSON2PG_DUMP << 1)
#   define JSON2PG_FDMP /* 0x000400 */ (JSON2PG_DUMP << 2)
#   define JSON2PG_COPY    0x001000
#   define JSON2PG_COPZ /* 0x002000 */ (JSON2PG_COPY << 1)
#   define JSON2PG_COPA /* 0x004000 */ (JSON2PG_COPY << 2)
#   define JSON2PG_ARRA    0x010000
#   define JSON2PG_NUMB    0x020000
    BLIN_flag     subflags;
    json2pg_path  chain   ;
} json2pg_cf;

static const char *jtnm[JSON2PG_TYNUM + 1] =
{ "null   "
, "boolean"
, "double "
, "int    "
, "object "
, "array  "
, "string "
, "unknown"
};

static u_char *
forcopy(json2pg_cf *options, const char *n) {
#   define blin_internal_flags (options->flags & BLIN_MASK)
    int     ex = EX_OK;
    u_char *m  ;
    char   *u  ;

    ifBLIN_QX2("+ |%s~", n);
    if  (!n) {
        m = NULL;
    } else if (!(m = (u_char*)(u = strdup(n)))) {
        ifBLIN_QW0("strdup %s", n);
    } else if ((ex = getCGIparmpass3(0, (const u_char *)"copy", &m))) {
        ifBLIN_QW0("getCGIparmpass3 %s", m);
        if  ((u_char*)u != m) free(u);
        free(m);
        m = NULL;
    } else {
        if  ((u_char*)u != m) free(u);
    }
    ifBLIN_QX2("- |%s~", m);
    return(m);
#   undef blin_internal_flags
}

static struct json_object *
json2pg_gopath(json2pg_cf *options, struct json_object *jobj, char **path) {
#   define blin_internal_flags (options->flags & BLIN_MASK)
    struct lh_entry    *entry;
    size_t              num  ;
    json_type           jt   ;

    ifBLIN_QX2("+ %zu |%s...", BLIN_I(jobj), !*path ? "" : *path);
    for (u_int y = 0; !!path[y]; ++y) {
        jt = json_object_get_type(jobj);
        ifBLIN_QX3("%s %s", jtnm[jt], path[y]);
        switch (jt) {
        case json_type_object:
            for (entry = json_object_get_object(jobj)->head; !!entry; entry = entry->next) {
                ifBLIN_QX3("%s", (const char *)entry->k);
                if  (!strcmp(path[y], (const char *)entry->k)) {
                    jobj = (struct json_object*)entry->v;
                    break;
            }   }
            break;
        case json_type_array:
            if  (!!path[y][strspn(path[y], "0123456789")]) {
                ifBLIN_QX0("Array index required, %s issued", path[y]);
                errno = EINVAL;
                goto out;
            }
            num = strtoul(path[y], NULL, 10);
            if  (num >= json_object_array_length(jobj)) {
                ifBLIN_QX0("Out of bounds %zu >= %zu", num, json_object_array_length(jobj));
                errno = EFAULT;
                goto out;
            }
            if  (!(jobj = json_object_array_get_idx(jobj, num))) {
                ifBLIN_QX0("json_object_array_get_idx");
                errno = EFAULT;
                goto out;
            }
            break;
        default:
            ifBLIN_QX0("Ill type: %s(%u)", jtnm[(JSON2PG_TYNUM < jt) ? JSON2PG_TYNUM + 1 : jt], jt);
            errno = ENOSYS;
            goto out;
    }   }
out:
    ifBLIN_QX2("- %zu", BLIN_I(jobj));
    return(jobj);
#   undef blin_internal_flags
}

static int
json2pg_table(json2pg_cf *options, struct json_object *jobj) {
#   define blin_internal_flags (options->flags & BLIN_MASK)
    json2pg_path             *path ;
    struct json_object       *jobn ;
    struct json_object       *val  ;
    u_int                     num  ;
    json_type                 jt   ;
    int                       ex   = EX_OK;
    int                       t    ; /* tab inform */

    ifBLIN_QX2("+ %zu", BLIN_I(jobj));
    if  (!(jobj = json2pg_gopath(options, jobj, options->chain.path))) {
        ifBLIN_QW0("json2pg_path");
        ex = EX_CONFIG;
        goto out;
    }
    if  (json_type_array != (jt = json_object_get_type(jobj))) {
        ifBLIN_QX0("Int path not points to array");
        ex = EX_USAGE;
        errno = ENOATTR;
        goto out;
    }
    for (num = 0; num < json_object_array_length(jobj); ++num) {
        if  (!(jobn = json_object_array_get_idx(jobj, num))) {
            ifBLIN_QX0("json_object_array_get_idx");
            errno = EFAULT;
            goto out;
        }
        t = 0;
        if  (JSON2PG_NUMB & options->flags) {
            printf("%u", num);
            t = 1;
        }
        for (path = options->chain.next; !!path; path = path->next) {
            ifBLIN_QX3("%s...", path->path[0]);
            if  (!(val = json2pg_gopath(options, jobn, path->path))) {
                printf("%s\\N", t ? "\t" : "");
                t = 1;
            } else {
                jt = json_object_get_type(val);
                ifBLIN_QX3("%s", jtnm[jt]);
                switch((JSON2PG_TYNUM < jt) ? JSON2PG_TYNUM + 1 : jt) {
                case json_type_null:
                    printf("%s\\N", t ? "\t" : "");
                    t = 1;
                    break;
                case json_type_boolean:
                    printf("%s%c", t ? "\t" : "", json_object_get_boolean(val) ? 't' : 'f');
                    t = 1;
                    break;
                case json_type_double:
                    printf("%s%g", t ? "\t" : "", json_object_get_double(val));
                    t = 1;
                    break;
                case json_type_int:
                    printf("%s%d", t ? "\t" : "", json_object_get_int(val));
                    t = 1;
                    break;
                case json_type_string: {
                        u_char *n;

                        if  (!(n = forcopy(options, json_object_get_string(val)))) {
                            ifBLIN_QW0("forcopy n %s", n);
                            ex = EX_OSERR;
                            goto out;
                        }
                        printf("%s%s", t ? "\t" : "", n);
                        free(n);
                    }
                    t = 1;
                    break;
                default:
                    ifBLIN_QX1( "%s(%d) is not element"
                              , jtnm[(JSON2PG_TYNUM < jt) ? JSON2PG_TYNUM + 1 : jt]
                              , jt
                              );
                    printf("%s\\N", t ? "\t" : "");
                    t = 1;
        }   }   }
        printf("\n");
    }
out:
    ifBLIN_QX2("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static u_char *
nmcopy(json2pg_cf *options, const char *n) {
#   define blin_internal_flags (options->flags & BLIN_MASK)
    int                         ex = EX_OK;
    union {u_char *u; char *d;} m  ;
    char                       *u  ;

    ifBLIN_QX2("+ |%s~", n);
    if  (!n) {
        m.u = NULL;
    } else if (JSON2PG_COPA & options->flags) {
        /* XXXX CTL-> \CTL, \ -> \\\\, " -> \\" */
        asprintf(&m.d, "\"%s\"", n);
    } else if (!(m.d = (u = strdup(n)))) {
        ifBLIN_QW0("strdup %s", n);
    } else if ((ex = getCGIparmpass3(0, (const u_char*)"copy", &m.u))) {
        ifBLIN_QW0("getCGIparmpass3 %s", m.d);
        if  (u != m.d) free(u);
        free(m.u);
        m.u = NULL;
    } else {
        if  (u != m.d) free(u);
    }
    ifBLIN_QX2("- |%s~", m.d);
    return(m.u);
#   undef blin_internal_flags
}

static int
json2pg_dump( json2pg_cf               *options
            , struct json_object       *jobj
            , const char              **pm     /*     */
            , const u_char             *nnm    /*                     */
            , int                       ex     /*        */
            , int                       exx    /*                   */
            ) {
#   define blin_internal_flags (options->flags & BLIN_MASK)
#   define IFNMM (nnmm ? nnmm : "\\N")
#   define IFNM (nnmm ? nnmm : "")
    struct lh_entry          *entry;
    u_int                     level; /*                                    */
    char                     *nnmm ; /*                       */
    const char               *key  ; /*     json_type_object  json_type_array */
    struct json_object       *val  ;
    u_char                   *pnm  ; /* COPY                 */
    u_char                   *jv   ; /*                                */
    json_type                 jt   ;

    if  (!pm) {
        ifBLIN_QX0("pm is NULL");
        ex = -EX_USAGE;
        goto out1;
    }
    ifBLIN_QX2( "+%08X %s:%s %u<-%u\n"
              , options->flags
              , *pm ? *pm : "()"
              , nnm ? (const char *)nnm : "()"
              , ex
              , exx
              )
    ;
    level = 0;
    nnmm = NULL;
    if  (!nnm) {
        ifBLIN_QX3(" 0 %s:()\n", *pm ? *pm : "()");
        pnm = (u_char*)strdup("\\N");
    } else if (!(pnm = nmcopy(options, (const char *)nnm))) {
        ifBLIN_QW0("nmcopy nnm");
        ex = -EX_OSERR;
        goto out;
    } else {
        ++level;
        if  (pm) {
            u_char      *ptnm;
            char        *tnm ;
            const char **ppm ;

            for (ppm = pm, nnmm = NULL; *ppm; ++ppm, ++level) {
                if  (!(ptnm = nmcopy(options, *ppm))) {
                    ifBLIN_QW0("nmcopy ppm");
                    ex = -EX_OSERR;
                    goto out;
                } else if (!nnmm) {
                    asprintf(&tnm, "%s", ptnm);
                } else if (JSON2PG_COPA & options->flags) {
                    asprintf(&tnm, "%s,%s", ptnm, nnmm);
                } else {
                    asprintf(&tnm, "%s.%s", ptnm, nnmm);
                }
                free(ptnm);
                ifBLIN_QX4("new0 =%s~\n", tnm);
                if  (!tnm) {
                    ifBLIN_QX0("no mem #1");
                    errno = ENOMEM;
                    ex = -EX_OSERR;
                    goto out;
                }
                if  (nnmm) free(nnmm);
                nnmm = tnm;
            }
            if  (JSON2PG_COPA & options->flags) {
                if  (nnmm) {
                    asprintf(&tnm, "{%s,%s}", nnmm, pnm);
                } else {
                    asprintf(&tnm, "{%s}", pnm);
                }
            } else if (nnmm) {
                asprintf(&tnm, "%s.%s", nnmm, pnm);
            } else {
                asprintf(&tnm, "%s", pnm);
            }
            free(nnmm);
            nnmm = tnm;
            ifBLIN_QX4("new1 =%s~\n", nnmm);
        } else if (nnm) {
            asprintf(&nnmm, "%s", pnm);
            ifBLIN_QX4("new2 =%s~\n", nnmm);
        }
        if  (!nnmm) {
            ifBLIN_QX0("no mem #2");
            errno = ENOMEM;
            ex = -EX_OSERR;
            goto out;
    }   }
    jt = json_object_get_type(jobj);
    switch (jt) {
    case json_type_null:
        if  (JSON2PG_COPZ & options->flags) {
            printf("%u	%u	%s	\\N\n", exx, jt, IFNMM);
        } else if (JSON2PG_COPY & options->flags) {
            printf("%u	%u	%u	%s	\\N\n", exx, ex, jt, pnm);
        } else {
            printf("%s %s\n", jtnm[jt], IFNM);
        }
        ex = exx;
        break;
    case json_type_boolean:
        if  (JSON2PG_COPZ & options->flags) {
            printf( "%u	%u	%s	%c\n"
                  , exx
                  , jt
                  , IFNMM
                  , json_object_get_boolean(jobj) ? 't' : 'f'
                  );
        } else if (JSON2PG_COPY & options->flags) {
            printf( "%u	%u	%u	%s	%c\n"
                  , exx
                  , ex
                  , jt
                  , pnm
                  , json_object_get_boolean(jobj) ? 't' : 'f'
                  );
        } else {
            printf("%s %s=%d\n", jtnm[jt], IFNM, json_object_get_boolean(jobj));
        }
        ex = exx;
        break;
    case json_type_double:
        if  (JSON2PG_COPZ & options->flags) {
            printf("%u	%u	%s	%g\n", exx, jt, IFNMM, json_object_get_double(jobj));
        } else if (JSON2PG_COPY & options->flags) {
            printf("%u	%u	%u	%s	%g\n", exx, ex, jt, pnm, json_object_get_double(jobj));
        } else {
            printf("%s %s=%g\n", jtnm[jt], IFNM, json_object_get_double(jobj));
        }
        ex = exx;
        break;
    case json_type_int:
        if  (JSON2PG_COPZ & options->flags) {
            printf("%u	%u	%s	%d\n", exx, jt, IFNMM, json_object_get_int(jobj));
        } else if (JSON2PG_COPY & options->flags) {
            printf("%u	%u	%u	%s	%d\n", exx, ex, jt, pnm, json_object_get_int(jobj));
        } else {
            printf("%s %s=%d\n", jtnm[jt], IFNM, json_object_get_int(jobj));
        }
        ex = exx;
        break;
    case json_type_object: {
        const char **pmm;
        u_int        i  ;

        if  (!(JSON2PG_DUMP & options->flags)) {
            jv = (u_char*)strdup("\\N");
        } else {
            if  (!(jv = forcopy(options, json_object_get_string(jobj)))) {
                ifBLIN_QW0("forcopy jv");
                ex = -EX_OSERR;
                goto out;
        }   }
        if  (JSON2PG_COPZ & options->flags) {
            printf("%u	%u	%s	%s\n", exx, jt, IFNMM, jv);
        } else if (JSON2PG_COPY & options->flags) {
            printf("%u	%u	%u	%s	%s\n", exx, ex, jt, pnm, jv);
        } else if (JSON2PG_FDMP & options->flags) {
            printf("%s %s=%s~\n", jtnm[jt], IFNM, jv);
        } else if (JSON2PG_ODMP & options->flags) {
            printf("%s %s\n", jtnm[jt], IFNM);
        }
        free(jv);
        ex = exx;
        pmm = malloc((1UL + level) * sizeof(pm));
        if  (!pmm) {
            ifBLIN_QX0("no mem #3");
            errno = ENOMEM;
            ex = -EX_OSERR;
            goto out;
        }
        pmm[0] = (const char*)nnm;
        for (i = 0; i < level; ++i) pmm[i + 1] = pm[i];
        for (entry = json_object_get_object(jobj)->head; (ex >= 0) && entry; entry = entry->next) {
            key = (const char*)entry->k;
            val = (struct json_object*)entry->v;
            exx = json2pg_dump(options, val, pmm, (const u_char*)key, ex, ++exx);
        }
        free(pmm);
        ex = exx;
        break;
    }
    case json_type_array: {
        const char **pmm;
        size_t       i  ;

        if  (!(JSON2PG_DUMP & options->flags)) {
            jv = (u_char*)strdup("\\N");
        } else {
            if  (!(jv = forcopy(options, json_object_get_string(jobj)))) {
                ifBLIN_QW0("forcopy jv");
                ex = -EX_OSERR;
                goto out;
        }   }
        if  (JSON2PG_COPZ & options->flags) {
            printf("%u	%u	%s	%s\n", exx, jt, IFNMM, jv);
        } else if (JSON2PG_COPY & options->flags) {
            printf("%u	%u	%u	%s	%s\n", exx, ex, jt, pnm, jv);
        } else if (JSON2PG_FDMP & options->flags) {
            printf("%s %s=%s~\n", jtnm[jt], IFNM, jv);
        } else if (JSON2PG_ODMP & options->flags) {
            printf("%s %s\n", jtnm[jt], IFNM);
        }
        ifBLIN_QX3("free jv %"BLIN_X"\n", BLIN_I(jv));
        free(jv);
        ex = exx;
        pmm = malloc((1 + level) * sizeof(pm));
        if  (!pmm) {
            ifBLIN_QX0("no mem #4");
            errno = ENOMEM;
            ex = -EX_OSERR;
            goto out;
        }
        pmm[0] = (const char *)nnm;
        for (i = 0; i < level; ++i) pmm[i + 1] = pm[i];
        for (i = 0; (ex >= 0) && (i < json_object_array_length(jobj)); ++i) {
            char *k;

            val = json_object_array_get_idx(jobj, i);
            asprintf(&k, "%zu", i);
            if  (!k) {
                ifBLIN_QW0("asprintf");
                ex = -EX_OSERR;
                goto out;
            }
            exx = json2pg_dump(options, val, pmm, (u_char *)k, ex, ++exx);
            free(k);
        }
        free(pmm);
        ex = exx;
        break;
    }
    case json_type_string:
        if  (JSON2PG_COPY & options->flags) {
            u_char *n;

            if  (!(n = forcopy(options, json_object_get_string(jobj)))) {
                ifBLIN_QW0("forcopy n %s", n);
                ex = -EX_OSERR;
                goto out;
            }
            if  (JSON2PG_COPZ & options->flags) {
                printf("%u	%u	%s	%s\n", exx, jt, IFNMM, n);
            } else {
                printf("%u	%u	%u	%s	%s\n", exx, ex, jt, pnm, n);
            }
            free(n);
        } else {
            printf("%s %s=%s~\n", jtnm[jt], IFNM, json_object_get_string(jobj));
        }
        ex = exx;
        break;
    default:
        if  (JSON2PG_COPY & options->flags) {
            u_char *n;

            if  (!(n = forcopy(options, strdup(json_object_get_string(jobj))))) {
                ifBLIN_QW0("forcopy n %s", n);
                ex = -EX_OSERR;
                goto out;
            }
            if  (JSON2PG_COPZ & options->flags) {
                printf("%u	%u	%s	%s\n", exx, jt, IFNMM, n);
            } else {
                printf("%u	%u	%u	%s	%s\n", exx, ex, jt, pnm, n);
            }
            free(n);
        } else {
            printf("%2d=unkn %s=%s~\n", jt, IFNM, json_object_get_string(jobj));
        }
        ex = exx;
        break;
    }
    free(nnmm);
out:
    free(pnm);
out1:
    ifBLIN_QX2("-%08X %u<-%u\n", options->flags, ex, exx);
    return(ex);
#   undef blin_internal_flags
}

static char **
restruct(json2pg_cf *options, const char *o) {
#   define blin_internal_flags (options->flags & BLIN_MASK)
    char       **res = NULL;
    const char  *q   ;
    const char  *d   ;
    u_int        p   ;

    if  (!o) {
        ifBLIN_QX0("Empty path");
        errno = EFAULT;
        goto out;
    }
    ifBLIN_QX2("+ |%s~", o);
    for (p = 2, q = o; !!(q = strchr(q, '.')); ++p) ++q;
    if  (!(res = calloc(p, sizeof(char *)))) {
        ifBLIN_QW0("");
        goto out;
    }
    q = o;
    for (u_int i = 0; ; ++i) {
        d = strchr(q, '.');
        if  (!d) {
            res[i] = strdup(q);
            break;
        } else {
            res[i] = strndup(q, (size_t)(d++ - q));
            q = d;
    }   }
out:
    ifBLIN_QX2("- %s...", *res);
    return(res);
#   undef blin_internal_flags
}

__attribute__((__noreturn__)) static void
usage(int ex) {
    int i;

    fprintf( stderr
           , "json2pg @BABOLO V.M "VERS"  "DATE" http://www.babolo.ru/\n"
             "    -a           - COPY series of objects\n"
             "    -b path      - array path to unroll to table\n"
             "    -c[c[c]]     - full output, COPY object recursive/plain\n"
             "    -D[D[D]]     - full output, dump object\n"
             "    -E -flag env - substitute argument of -flag by environment variable value\n"
             "    -i path      - element path to column for output table. Column order as -i order\n"
             "    -n           - cortege number in first column when unroll to table\n"
             "    -q           - quiet\n"
             "    -v           - verbose\n"
             "    -?           - this text\n"
             " Columns in full output are:\n"
             "   - num        (when -c[c[c]])\n"
             "   - parent num (when -c, not -cc or -ccc)\n"
             "   - type\n"
             "   - name       (when -c[c[c]])\n"
             "   - value      (when -c[c[c]], full value with -D)\n"
             "   - name=value (when -D without -c[c[c]])\n"
             " List of types:\n"
           )
    ;
    for (i = 0; i < JSON2PG_TYNUM; ++i) {
        fprintf(stderr, "  %2d %s\n", i, jtnm[i]);
    }
    exit(ex);
}

int
main(int argc, char *argv[]) {
    json2pg_cf          *options;
    off_t                joffset;
    ssize_t              jsize  ;
    json_object         *jobj   ;
    int                  jnum   ;
    const char          *flar   = "ab:cDi:nqv=:?";
    json2pg_path        *path   ;
    const char          *nnm    = NULL;
    const char         **pmm    = &nnm;
    babolo_opts         *bos    = NULL;
    struct json_tokener *tok    ;
    mife_descriptor     *mdi    ;
    int                  ex     = EX_OK;
    const char          *o      ;
    int                  c      ;
    u_int                k      ;

    if  (!(options = calloc(1, sizeof(*options)))) {
        warnx("Malloc failed #1");
        errno = ENOMEM;
        ex = EX_OSERR;
        goto out;
    }
    blin_ctl(BLIN_CTL_FLAG | BLIN_CTL_FSET, (options->flags = BLIN_BIT0));
#   define blin_internal_flags (options->flags & BLIN_MASK)
    if  (!(bos = babolo_openopts(Bpars_NONU | Bpars_SYME, 'Z'))) {
        ifBLIN_QW0("babolo_openopts");
        ex = EX_SOFTWARE;
        goto out;
    }
    if  (babolo_setopts(bos, Bpars_NONU, argc, argv, flar)) {
        ifBLIN_QW0("babolo_setopts");
        ex = EX_SOFTWARE;
        goto out;
    }
    while (0 < (c = babolo_getopts(bos))) {
        switch (c) {
        case 'a': options->flags |= JSON2PG_ARRA;
                ; break;
        case 'b': if  (!(options->chain.path = restruct(options, (o = babolo_getoptsarg(bos))))) {
                ;     ifBLIN_QW0("restruct -%c %s", bos->c, o);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; }
                ; break;
        case 'c': options->flags |= ((JSON2PG_COPY | JSON2PG_COPZ) & options->flags) << 1;
                ; options->flags |= JSON2PG_COPY;
                ; break;
        case 'D': options->flags |= ((JSON2PG_DUMP | JSON2PG_ODMP) & options->flags) << 1;
                ; options->flags |= JSON2PG_DUMP;
                ; break;
        case 'i': path = &options->chain;
                ; for (k = 0; !!path->next; path = path->next, k++) {}
                ; if  (!(path->next = calloc(1, sizeof(json2pg_path)))) {
                ;     ifBLIN_QW0("No mem");
                ;     ex = EX_OSERR;
                ;     goto out;
                ; }
                ; if  (!(path->next->path = restruct(options, (o = babolo_getoptsarg(bos))))) {
                ;     ifBLIN_QW0("restruct @%u -%c %s", k, bos->c, o);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; }
                ; break;
        case 'n': options->flags |= JSON2PG_NUMB;
                ; break;
        case 'q': options->flags &= ~BLIN_MASK;
                ; break;
        case 'v': BLIN_VERBOSE(options->flags);
                ; break;
        case '=': if  (!(o = babolo_getoptsarg(bos))) {
                ;     ifBLIN_QW0("No -%c arg", bos->c);
                ;     ex = EX_USAGE;
                ;     goto out;
                ; }
                ; switch(o[0]) {
                  case '0': blin_ctl(BLIN_CTL_DUMP);           exit(EX_OK);
                  case '1': babolo_optest(flar);               exit(EX_OK);
                  case '2': babolo_optext(bos);                exit(EX_OK);
                  case '3': babolo_dumpopts(bos);              exit(EX_OK);
                  }
                ; /* FALLTHROUGH */
        case '?': usage(EX_OK);
        default : ifBLIN_QX0("Illegal flag -%c", c);
                ; usage(EX_USAGE);
    }   }
    options->subflags = MIFE_SLID
                      | ( BLIN_MASK
                        & ((BLIN_GEN1 & options->flags) | (options->flags >> 3)
                        ) )
    ;
    if  (!(mdi = mife_init(options->subflags))) {
        ifBLIN_QW0("mife_init stdin");
        ex = EX_IOERR;
        goto out;
    }
    if  (!(o = babolo_getargs(bos))) {
        if  (0 > mife_ctlfdsc(mdi, fileno(stdin))) {
            ifBLIN_QW0("mife_ctlfdsc stdin");
            ex = EX_IOERR;
            goto out;
        }
    } else {
        if  (0 > mife_ctlfile(mdi, o)) {
            ifBLIN_QW0("mife_ctlfile %s", o);
            ex = EX_IOERR;
            goto out;
    }   }
    if  (!(tok = json_tokener_new())) {
        ifBLIN_QW0("json_tokener_new");
        errno = EDOOFUS;
        ex = EX_SOFTWARE;
        goto out;
    }
    joffset = 0;
    jnum = 0;
    for (int i = 0; ; ++i) {
        do {
            if  (0 > (jsize = mife_read(mdi, 0, joffset))) {
                ifBLIN_QW0("mife_read stdin");
                ex = EX_IOERR;
                goto out;
            }
            if  (!jsize) {
                ifBLIN_QX1("null stdin");
                goto out;
            }
            if  (jsize > INT_MAX) {
                ifBLIN_QX0("Huge stdin out of possibility");
                ex = EX_UNAVAILABLE;
                goto out;
            }
            if  (!(o = mife_get(mdi, joffset))) {
                ifBLIN_QW0("mife_get stdin");
                ex = EX_IOERR;
                goto out;
           }
           jobj = json_tokener_parse_ex(tok, o, (int)jsize);
           joffset += tok->char_offset;
        } while(json_tokener_continue == tok->err);
        if  (json_tokener_success != tok->err) {
            ifBLIN_QX0("json_tokener_parse_ex: %d=%s", tok->err, json_tokener_error_desc(tok->err));
            errno = EDOOFUS;
            ex = EX_SOFTWARE;
            goto out;
        } else if (!jobj) {
            ifBLIN_QX0("json_tokener_parse_ex: unknown error");
            errno = EDOOFUS;
            ex = EX_SOFTWARE;
            goto out;
        }
        if (tok->char_offset < jsize) {
            ifBLIN_QX1("Extra symbols=%s~", o + tok->char_offset);
        }
        if  ((JSON2PG_DUMP | JSON2PG_COPY) & options->flags) {
            if  (0 > (jnum = json2pg_dump(options, jobj, pmm, NULL, 0, ++jnum))) {
                ifBLIN_QX0("json2pg_dump");
                ex = -jnum;
                goto out;
            } else {
                ex = 0;
            }
        } else if (options->chain.path) {
            if  (!!(ex = json2pg_table(options, jobj))) {
                ifBLIN_QW0("json2pg_table");
                goto out;
        }   }
        if  (tok->char_offset >= jsize) break;
        if  (JSON2PG_ARRA & options->flags) continue;
        ifBLIN_QX1("Extra symbols=%s~", o + tok->char_offset);
        break;
    }
out:
    exit(ex);
#   undef blin_internal_flags
}
